//===========================================================================
// InformStyleRecord.cpp	Implementation of the Inform Style display props
//===========================================================================

#include "stdafx.h"
#include "InformStyleRecord.h"


//***************************************************************************
// InformStyleRecord
//***************************************************************************

/////////////////////////////////////////////////////////////////////////////
// Static Data
//
const LPCTSTR InformStyleRecord::s_SectionName = 
	_T("Inform Syntax Colours");


/////////////////////////////////////////////////////////////////////////////
// Construction
//
InformStyleRecord::InformStyleRecord(LPCTSTR name, InformStyle id) :
	m_Name(name),
	m_Id(id)
{
	ASSERT(!m_Name.IsEmpty());
}



/////////////////////////////////////////////////////////////////////////////
// operator=()
//
InformStyleRecord& 
InformStyleRecord::operator=(const InformStyleRecord& src)
{
	m_Id = src.m_Id;
	m_Name = src.m_Name;
	Colour(src.Colour());
	Bold(src.Bold());
	Italic(src.Italic());
	return *this;
}


/////////////////////////////////////////////////////////////////////////////
// operator==()
//
BOOL InformStyleRecord::operator==(const InformStyleRecord& other)
{
	return m_Id == other.m_Id && 
		   m_Name == other.m_Name &&
		   Colour() == other.Colour() && 
		   Bold() == other.Bold() && 
		   Italic() == other.Italic();
}


/////////////////////////////////////////////////////////////////////////////
// Read()
//
// Read the display properties for this syntax colour from the .INI file or 
// registry.  If these can't be found, use the given defaults.
//
void InformStyleRecord::Read(COLORREF colour /*=RGB(0,0,0)*/, 
									BOOL bold /*=FALSE*/, 
									BOOL italic /*=FALSE*/)
{
	CWinApp* pApp = AfxGetApp();
	ASSERT(pApp);

	CString item(m_Name + _T(" Colour"));
	Colour(COLORREF(pApp->GetProfileInt(s_SectionName, item, colour)));

	item = m_Name + _T(" Bold");
	Bold(pApp->GetProfileInt(s_SectionName, item, bold) ? TRUE : FALSE);

	item = m_Name + _T(" Italic");
	Italic(pApp->GetProfileInt(s_SectionName, item, italic) ? TRUE : FALSE);
}


/////////////////////////////////////////////////////////////////////////////
// Write()
//
// Write the current display properties for this syntax colour to the .INI 
// file or registry.  Ignores error since it's not clear what we should do 
// about them.
//
void InformStyleRecord::Write()
{
	CWinApp* pApp = AfxGetApp();
	ASSERT(pApp);

	CString item(m_Name + _T(" Colour"));
	pApp->WriteProfileInt(s_SectionName, item, int(Colour()));

	item = m_Name + _T(" Bold");
	pApp->WriteProfileInt(s_SectionName, item, Bold());

	item = m_Name + _T(" Italic");
	pApp->WriteProfileInt(s_SectionName, item, Italic());
}


//***************************************************************************
// InformStyleRecordSet
//***************************************************************************

/////////////////////////////////////////////////////////////////////////////
// THE master style records
//
InformStyleRecordSet theMasterStyles;


/////////////////////////////////////////////////////////////////////////////
// operator=()
//
InformStyleRecordSet& 
InformStyleRecordSet::operator=(const InformStyleRecordSet& src)
{
	DeleteContents();
	POSITION pos = src.GetStartPosition();
	InformStyle key = InvalidStyle;
	InformStyleRecord * entry;
	while ( NULL != pos )
	{
		src.GetNextAssoc(pos, key, entry);
		ASSERT(entry);
		SetAt(key, new InformStyleRecord(*entry));
	}
	return *this;
}


/////////////////////////////////////////////////////////////////////////////
// operator==()
//
BOOL InformStyleRecordSet::operator==(const InformStyleRecordSet& other)
{
	if ( GetCount() != other.GetCount() )
		return FALSE;
	POSITION pos = other.GetStartPosition();
	InformStyle key = InvalidStyle;
	InformStyleRecord * entry, *otherEntry;
	while ( NULL != pos )
	{
		other.GetNextAssoc(pos, key, otherEntry);
		if ( !Lookup(key, entry) || *entry != *otherEntry )
			return FALSE;
	}
	return TRUE;
}


/////////////////////////////////////////////////////////////////////////////
// DeleteContents()
//
// Get rid of all style records in this set.
//
void InformStyleRecordSet::DeleteContents()
{
	POSITION pos = GetStartPosition();
	InformStyle key = InvalidStyle;
	InformStyleRecord * entry;
	while ( NULL != pos )
	{
		GetNextAssoc(pos, key, entry);
		delete entry;
	}
	RemoveAll();
}


/////////////////////////////////////////////////////////////////////////////
// LoadDefaults()
//
// Set up the standard set of style records and try to read their attributes 
// from the registry, using values similar to Graham's if not.
//
void InformStyleRecordSet::LoadDefaults()
{
	DeleteContents();
	InformStyleRecord * isc;

	isc = new InformStyleRecord(_T("Foreground"), ForegroundStyle);
	isc->Read(RGB(0x00,0x00,0xA0));
	SetAt(ForegroundStyle, isc);
	isc = new InformStyleRecord(_T("Quoted Text"), QuotedTextStyle);
	isc->Read(RGB(0x80,0x80,0x80));
	SetAt(QuotedTextStyle, isc);
	isc = new InformStyleRecord(_T("Comment"), CommentStyle);
	isc->Read(RGB(0x80,0x00,0x80), FALSE, TRUE);
	SetAt(CommentStyle, isc);
	isc = new InformStyleRecord(_T("Directive"), DirectiveStyle);
	isc->Read(RGB(0x00,0x00,0x00), TRUE);
	SetAt(DirectiveStyle, isc);
	isc = new InformStyleRecord(_T("Property"), PropertyStyle);
	isc->Read(RGB(0xC0,0x00,0x00));
	SetAt(PropertyStyle, isc);
	isc = new InformStyleRecord(_T("Function"), FunctionStyle);
	isc->Read(RGB(0xA0,0x00,0x00));
	SetAt(FunctionStyle, isc);
	isc = new InformStyleRecord(_T("Code"), CodeStyle);
	isc->Read(RGB(0x00,0x00,0x80));
	SetAt(CodeStyle, isc);
	isc = new InformStyleRecord(_T("Code Alpha"), CodeAlphaStyle);
	isc->Read(RGB(0x00,0x80,0x00));
	SetAt(CodeAlphaStyle, isc);
	isc = new InformStyleRecord(_T("Assembly"), AssemblyStyle);
	isc->Read(RGB(0xA0,0xA0,0x00), FALSE, TRUE);
	SetAt(AssemblyStyle, isc);
	isc = new InformStyleRecord(_T("Escape Character"), EscapeStyle);
	isc->Read(RGB(0xC0,0x00,0x00));
	SetAt(EscapeStyle, isc);

	isc = new InformStyleRecord(_T("Number"), NumberStyle);
	isc->Read(RGB(0x00,0x00,0xA0), TRUE);
	SetAt(NumberStyle, isc);
	isc = new InformStyleRecord(_T("Code Number"), CodeNumberStyle);
	isc->Read(RGB(0x00,0x80,0x00), TRUE);
	SetAt(CodeNumberStyle, isc);
}


/////////////////////////////////////////////////////////////////////////////
// SaveDefaults()
//
// Save the style records to the registry.
//
void InformStyleRecordSet::SaveDefaults()
{
	POSITION pos = GetStartPosition();
	InformStyle key = InvalidStyle;
	InformStyleRecord * entry;
	while ( NULL != pos )
	{
		GetNextAssoc(pos, key, entry);
		ASSERT(entry);
		entry->Write();
	}
}

