//===========================================================================
// InformStyleRecord.h	Definition of the Inform Style display properties
//
// This maintains a record of the colour, bold and italic settings to use 
// when representing a particular syntax style on screen.  Each style also 
// has a name (defined when the record is created).  Methods exist to write 
// this information to the registry (and read it back), to copy records, and 
// to compare them.
//
// The records are grouped together into record sets, mapped by syntax style.
// There is assumed to be one master set, accessible via GetMasterStyles();
// calling this set's LoadDefaults() and SaveDefaults() methods is all that 
// is needed to keep the registry up-to-date.
//===========================================================================

#if !defined(INFORMSTYLERECORD_H)
#define INFORMSTYLERECORD_H


//***************************************************************************
// Includes
//***************************************************************************

#include "InformStyle.h"
#include <afxtempl.h>


//***************************************************************************
// Definitions
//***************************************************************************

/////////////////////////////////////////////////////////////////////////////
// InformStyleRecord Definition
//
class InformStyleRecord
{
// Interface
  public:
	InformStyleRecord(LPCTSTR name, InformStyle id);
	InformStyleRecord(const InformStyleRecord& src)
		{ *this = src; }
	InformStyleRecord& operator=(const InformStyleRecord& src);

	~InformStyleRecord()			{ }

	BOOL operator==(const InformStyleRecord& other);
	BOOL operator!=(const InformStyleRecord& other)
		{ return !(*this == other); }

	InformStyle	Id() const				{ return m_Id; }
	LPCTSTR		Name() const			{ return m_Name; }
	COLORREF	Colour() const			{ return m_Colour; }
	BOOL		Bold() const			{ return m_Bold; }
	BOOL		Italic() const			{ return m_Italic; }

	void		Colour(COLORREF colour)	{ m_Colour = colour; }
	void		Bold(BOOL bold)			{ m_Bold = bold; }
	void		Italic(BOOL italic)		{ m_Italic = italic; }

	void		Read(COLORREF colour=RGB(0,0,0), BOOL bold=FALSE, 
					 BOOL italic=FALSE);
	void		Write();

// Implementation
  private:
	InformStyle	m_Id;
	CString 	m_Name;
	COLORREF	m_Colour;
	BOOL		m_Bold;
	BOOL		m_Italic;
	static const LPCTSTR	s_SectionName;
};


/////////////////////////////////////////////////////////////////////////////
// InformStyleRecordSet Definition
//
class InformStyleRecordSet : 
	public CTypedPtrMap<CMapWordToPtr, InformStyle, InformStyleRecord*>
{
// Interface
  public:
	InformStyleRecordSet()		{}
	InformStyleRecordSet(const InformStyleRecordSet& src)
		{ *this = src; }
	InformStyleRecordSet& operator=(const InformStyleRecordSet& src);
	~InformStyleRecordSet()		{ DeleteContents(); }

	BOOL operator==(const InformStyleRecordSet& other);
	BOOL operator!=(const InformStyleRecordSet& other)
		{ return !(*this == other); }
	void		DeleteContents();
	void		LoadDefaults();
	void		SaveDefaults();

	static InformStyleRecordSet &	GetMasterStyles();
};

//---------------------------------------------------------------------------

extern InformStyleRecordSet theMasterStyles;
inline InformStyleRecordSet &	InformStyleRecordSet::GetMasterStyles()
		{ return theMasterStyles; }

#endif //!defined(INFORMSTYLERECORD_H)

