//===========================================================================
// InformStyler.cpp		Implementation of the Inform Syntax Styler
//===========================================================================

#include "stdafx.h"
#include "InformStyler.h"
#include <string.h>


//***************************************************************************
// inline implementation methods
//***************************************************************************

/////////////////////////////////////////////////////////////////////////////
// Includes()
//
// Test to see if the state includes ALL the requested flags.
//
inline BOOL InformStyler::Includes(DWORD mask) const
{
	return (mask & m_State) == mask;
}


/////////////////////////////////////////////////////////////////////////////
// Set(), Clear()
//
// Add or remove the requested flags.
//
inline void InformStyler::Set(DWORD mask)
{
	m_State |= mask;
}

inline void InformStyler::Clear(DWORD mask)
{
	m_State &= ~mask;
}


/////////////////////////////////////////////////////////////////////////////
// IsKeywordTerminal(), IsPointer(), IsAsterisk(), IsTerminal()
//
// Check inner state for various conditions.
//
inline BOOL InformStyler::IsKeywordTerminal() const
{
	return 0x8100 <= (0xFFFF & m_State);
}

inline BOOL InformStyler::IsPointer() const
{
	return 0x0002 == (0xFFFF & m_State);
}

inline BOOL InformStyler::IsAsterisk() const
{
	return 0x0003 == (0xFFFF & m_State);
}

inline BOOL InformStyler::IsTerminal() const
{
	return IsKeywordTerminal() && IsPointer() && IsAsterisk();
}


/////////////////////////////////////////////////////////////////////////////
// InnerState()
//
// Get the inner state.
//
inline WORD InformStyler::InnerState() const
{
	return (WORD)(0xFFFF & m_State);
}


/////////////////////////////////////////////////////////////////////////////
// ClearInnerState(), SetInnerState()
//
// Change the inner state.
//
inline void InformStyler::ClearInnerState()
{
	m_State &= ~0xFFFF;
}

inline void InformStyler::SetInnerState(WORD newIS)
{
	m_State = (~0xFFFF & m_State) | newIS;
}


	
//***************************************************************************
// Interface
//***************************************************************************

/////////////////////////////////////////////////////////////////////////////
// ColourString()
//
// Create a styling character map of the string (based on the state prior to 
// the start of the text) and return the final state.  Perform all of phases
// (b)-(d) as described by Graham Nelson (see "Algorithms.txt"), except for 
// running the inner state machine - see also RunInnerState().
//
// Assume that map points to a buffer at least as long as the text, 
// *including* the terminating null.  The terminator for the map will be an 
// InvalidStyle entry.
//
DWORD InformStyler::ColourString(LPCTSTR text, unsigned char * map)
{
	ASSERT(map);
	ASSERT(text);

	InformStyle backtrackStyle;

	size_t textLength = strlen(text);

	// Phases (b) Scanning text and (c) Initial colouring combined:
	size_t pos;
	for ( pos = 0 ; textLength > pos ; pos++ )
	{
		// Run the state machine as described in Graham Nelson's docs,
		// but where possible colour as we go:
		if ( Includes(CommentFlag) )			// 1.
		{
			map[pos] = (unsigned char)CommentStyle;
			if ( '\n' == text[pos] )
				Clear(CommentFlag);
		}
		else if ( Includes(DoubleQuoteFlag) )	// 2.
		{
			map[pos] = (unsigned char)QuotedTextStyle;
			if ( '\"' == text[pos] )
				Clear(DoubleQuoteFlag);
		}
		else if ( Includes(SingleQuoteFlag) )	// 3.
		{
			map[pos] = (unsigned char)QuotedTextStyle;
			if ( '\'' == text[pos] )
				Clear(SingleQuoteFlag);
		}
		else if ( '\'' == text[pos] )			// 4.
		{
			map[pos] = (unsigned char)QuotedTextStyle;
			Set(SingleQuoteFlag);
		}
		else if ( '\"' == text[pos] )			// 5.
		{
			map[pos] = (unsigned char)QuotedTextStyle;
			Set(DoubleQuoteFlag);
		}
		else if ( '!' == text[pos] )			// 6.
		{
			map[pos] = (unsigned char)CommentStyle;
			Set(CommentFlag);
		}
		else if ( Includes(StatementFlag) )		// 7.
		{
			if ( ']' == text[pos] )
			{
				map[pos] = (unsigned char)FunctionStyle;
				Clear(StatementFlag);
			}
			else if ( isspace(text[pos]) )
			{
				map[pos] = (unsigned char)CodeStyle;
			}
			else if ( Includes(AfterRestartFlag) )
			{
				size_t colourPos = pos;
				RunInnerState(text, textLength, pos);

				if ( IsKeywordTerminal() )
				{
					Clear(AfterRestartFlag);
					Set(ColourBacktrackFlag);
					backtrackStyle = FunctionStyle;

					while ( pos >= colourPos )
						map[colourPos++] = (unsigned char)backtrackStyle;
					Clear(ColourBacktrackFlag);
				}
				else
					map[pos] = (unsigned char)CodeStyle;
			}
			else
				map[pos] = (unsigned char)CodeStyle;
		}
		else
		{
			if ( '[' == text[pos] )
			{
				map[pos] = (unsigned char)FunctionStyle;
				Set(StatementFlag);
				if ( !Includes(AfterMarkerFlag) )	//##Mod
					Set(AfterRestartFlag);
			}
			else if ( isspace(text[pos]) )
			{
				map[pos] = (unsigned char)ForegroundStyle;
			}
			else
			{
				size_t colourPos = pos;
				RunInnerState(text, textLength, pos);

				switch ( InnerState() )
				{
				  case 2:	//"->"
				  case 3:	//"*"
				  {
					Set(AfterMarkerFlag);
					Set(ColourBacktrackFlag);
					backtrackStyle = DirectiveStyle;
					break;
				  }
				  case 0x8404:	//"with"
				  {
					Set(AfterMarkerFlag);
					Set(ColourBacktrackFlag);
					backtrackStyle = DirectiveStyle;
					Set(HighlightFlag);
					Clear(HighlightAllFlag);
					break;
				  }
				  case 0x8313:	//"has"
				  case 0x8525:	//"class"
				  {
					Set(AfterMarkerFlag);
					Set(ColourBacktrackFlag);
					backtrackStyle = DirectiveStyle;
					Clear(HighlightFlag);
					Set(HighlightAllFlag);
					break;
				  }
				  default:
				  {
					if ( Includes(WaitDirectiveFlag) )
					{
						Clear(WaitDirectiveFlag);
						Set(ColourBacktrackFlag);
						backtrackStyle = DirectiveStyle;
					}
					else if ( Includes(HighlightAllFlag) )
					{
						Set(ColourBacktrackFlag);
						backtrackStyle = PropertyStyle;
					}
					else if ( Includes(HighlightFlag) )
					{
						Clear(HighlightFlag);
						Set(ColourBacktrackFlag);
						backtrackStyle = PropertyStyle;
					}
					else
						backtrackStyle = ForegroundStyle;
					break;
				  }
				}

				while ( pos >= colourPos )
					map[colourPos++] = (unsigned char)backtrackStyle;
				Clear(ColourBacktrackFlag);

				if ( ';' == text[pos] )
				{
					map[pos] = (unsigned char)DirectiveStyle;
					Set(WaitDirectiveFlag);
					Clear(AfterMarkerFlag);
					Clear(AfterRestartFlag);
					Clear(HighlightFlag);
					Clear(HighlightAllFlag);
				}
				else if ( ',' == text[pos] )
				{
					map[pos] = (unsigned char)DirectiveStyle;
					Set(AfterMarkerFlag);
					Set(HighlightFlag);
				}
				else
					map[pos] = (unsigned char)backtrackStyle;
			}
		}
	}

	// Phase (d) Colour refinement:
	for ( pos = 0 ; textLength > pos ; pos++ )
	{
		unsigned char currentStyle = map[pos];
		if ( QuotedTextStyle == currentStyle )
		{
			switch(text[pos])
			{
			  case '~':
			  case '^':
			  case '\\':
			  {
				map[pos] = EscapeStyle;
				break;
			  }
			  case '@':
			  {
				do
				{
					map[pos++] = EscapeStyle;
				}
				while ( textLength > pos && 
						( '@' == text[pos] || isdigit(text[pos]) ) );
				pos--;
				break;
			  }
			}
		}
		else if ( ForegroundStyle == currentStyle || CodeStyle == currentStyle )
		{
			if ( '@' == text[pos] )
			{
				do
				{
					map[pos++] = AssemblyStyle;
				}
				while ( textLength > pos && isisym(text[pos]) );
				pos--;
			}
			else if ( '$' == text[pos] )	//##Mod Hex or binary constant
			{
				currentStyle = ( ForegroundStyle == currentStyle ) ? 
									NumberStyle : CodeNumberStyle;
				map[pos++] = currentStyle;
				if ( textLength > pos )
				{
					if ( '$' == text[pos] )	// binary
					{
						map[pos++] = currentStyle;
						while ( textLength > pos && 
								( '0' == text[pos] || '1' == text[pos] ) )
							map[pos++] = currentStyle;
					}
					else						// hexadecimal
					{
						while ( textLength > pos && isxdigit(text[pos]) )
							map[pos++] = currentStyle;
					}
				}
				pos--;
			}
			else if ( isdigit(text[pos]) )		//##Mod decimal constant
			{
				currentStyle = ( ForegroundStyle == currentStyle ) ? 
									NumberStyle : CodeNumberStyle;
				do
				{
					map[pos++] = currentStyle;
				}
				while ( textLength > pos && isdigit(text[pos]) );
				pos--;
			}
			else if ( isisymf(text[pos]) )
			{
				if ( ForegroundStyle == currentStyle )
				{
					if ( MatchesKeyword(s_ForegroundKeys, &text[pos]) )
					{
						while ( textLength > pos && isisym(text[pos]) )
							map[pos++] = DirectiveStyle;
					}
					else
					{
						while ( textLength > pos && isisym(text[pos]) )
							pos++;
					}
				}
				else
				{
					if ( !MatchesKeyword(s_CodeKeys, &text[pos]) )
					{
						while ( textLength > pos && isisym(text[pos]) )
							map[pos++] = CodeAlphaStyle;
					}
					else
					{
						while ( textLength > pos && isisym(text[pos]) )
							pos++;
					}
				}
				pos--;
			}
		}
	}

	// Terminate map and return final state.
	map[textLength] = (unsigned char)InvalidStyle;
	return m_State;
}


//***************************************************************************
// Implementation
//***************************************************************************

/////////////////////////////////////////////////////////////////////////////
// Data
//
const LPCTSTR InformStyler::s_CodeKeys[] =
{
	_T("box"),  	_T("break"),	_T("child"), 	_T("children"), 
	_T("continue"), _T("default"),	_T("do"),   	_T("elder"), 	
	_T("eldest"), 	_T("else"), 	_T("false"), 	_T("font"),
	_T("for"),  	_T("give"), 	_T("has"),  	_T("hasnt"), 	
	_T("if"),   	_T("in"),		_T("indirect"), _T("inversion"),
	_T("jump"), 	_T("metaclass"),_T("move"), 	_T("new_line"),
	_T("nothing"), 	_T("notin"), 	_T("objectloop"),_T("ofclass"), 
	_T("or"),   	_T("parent"),	_T("print"), 	_T("print_ret"),
	_T("provides"), _T("quit"), 	_T("random"), 	_T("read"),
	_T("remove"), 	_T("restore"), 	_T("return"), 	_T("rfalse"), 	
	_T("rtrue"), 	_T("save"), 	_T("sibling"), 	_T("spaces"), 	
	_T("string"), 	_T("style"), 	_T("switch"), 	_T("to"),
	_T("true"), 	_T("until"), 	_T("while"), 	_T("younger"), 	
	_T("youngest"), NULL
};

const LPCTSTR InformStyler::s_ForegroundKeys[] =
{
	_T("first"), 	_T("last"), 	_T("meta"),		_T("only"),		
	_T("private"), 	_T("replace"),	_T("reverse"),	_T("string"),	
	_T("table"), 	NULL
};


/////////////////////////////////////////////////////////////////////////////
// MatchesKeyword()
//
// Check whether the Inform symbol at text is in the table.
//
BOOL InformStyler::MatchesKeyword(const LPCTSTR * table, LPCTSTR text)
{
	ASSERT(table);
	ASSERT(text);

	for( ; *table ; table++ )
	{
		int length = strlen(*table);
		if ( !strncmp(*table, text, length) )
		{
			if ( !isisym(text[length]) )
				return TRUE;
		}
	}
	return FALSE;
}


/////////////////////////////////////////////////////////////////////////////
// RunInnerState()
//
// Run the inner state machine *mostly* as described by Graham Nelson (see 
// "Algorithms.txt").  Will often loop until it reaches a terminal.
//
void InformStyler::RunInnerState(LPCTSTR text, size_t textLength, 
								 size_t & pos)
{
	ASSERT(text);
	ClearInnerState();
	for ( ; textLength > pos ; pos++ )
	{
		switch ( InnerState() )
		{
		  case 0:
		  {
			switch (tolower(text[pos]))
			{
			  case '-':
			  {
				SetInnerState(1);
				break;
			  }
			  case '*':
			  {
				SetInnerState(3);
				return;
			  }
			  case 'w':
			  {
				SetInnerState(0x101);
				break;
			  }
			  case 'h':
			  {
				SetInnerState(0x111);
				break;
			  }
			  case 'c':
			  {
				SetInnerState(0x121);
				break;
			  }
			  case '#':
				return;
			  default:
			  {
				if ( !isspace(text[pos]) )
				{
					if ( isisym(text[pos]) )
					{
						SetInnerState(0x100);
					}
					else
					{
						SetInnerState(0xFF);
						return;
					}
				}
				break;
			  }
			}
			break;
		  }
		  case 1:
		  {
			if ( '>' == text[pos] )
			{
				SetInnerState(2);
				return;
			}
			else
			{
				SetInnerState(0xFF);
				pos--;
				return;
			}
			break;
		  }
		  case 2:
		  case 3:
			break;
		  case 0xFF:
		  {
			if ( isspace(text[pos]) )
				ClearInnerState();
			break;
		  }
		  default:
		  {
			if ( 0x100 > InnerState() )
				ASSERT(FALSE);
			else if ( 0x8000 <= InnerState() )
				ClearInnerState();
			else if ( !isisym(text[pos]) )
			{
				SetInnerState(0x8000 | InnerState());
				pos--;
				return;
			}
			else
			{
				char c = tolower(text[pos]);
				switch ( InnerState() )
				{
				  case 0x101: SetInnerState( ('i'==c) ? 0x202 : 0x200 ); break;
				  case 0x202: SetInnerState( ('t'==c) ? 0x303 : 0x300 ); break;
				  case 0x303: SetInnerState( ('h'==c) ? 0x404 : 0x400 ); break;
				  case 0x111: SetInnerState( ('a'==c) ? 0x212 : 0x200 ); break;
				  case 0x212: SetInnerState( ('s'==c) ? 0x313 : 0x300 ); break;
				  case 0x121: SetInnerState( ('l'==c) ? 0x222 : 0x200 ); break;
				  case 0x222: SetInnerState( ('a'==c) ? 0x323 : 0x300 ); break;
				  case 0x323: SetInnerState( ('s'==c) ? 0x424 : 0x400 ); break;
				  case 0x424: SetInnerState( ('s'==c) ? 0x525 : 0x500 ); break;
				  default: SetInnerState(0x100 + InnerState()); break;
				}
			}
			break;
		  }
		}
	}
}


