//===========================================================================
// InformStyler.h		Definition of the Inform Syntax Styler
//
// This class encapsulates the state machine from Graham Nelson's algorithm 
// (modified as described in "Algorithm.txt").  Its only significant public 
// method is ColourString(), which builds up a map of styles for a given text 
// string.  ColourString() is not optimised for speed, but it takes less than 
// a second on a P166 to process a 4400+ line Inform file one line at a time.
// ColourString() returns the final state for convenience, but it is also 
// accessible via GetState().
//
// I have made two static methods public as well: isisym() and isisymf(), 
// which are analagous to iscsym() and iscsymf().
//
// The internal workings are described in more detail in "InformStyler.cpp".
//===========================================================================

#if !defined(INFORMSTYLER_H)
#define INFORMSTYLER_H


//***************************************************************************
// Includes
//***************************************************************************

#include "InformStyle.h"
#include <ctype.h>


//***************************************************************************
// Definitions
//***************************************************************************

/////////////////////////////////////////////////////////////////////////////
// InformStyler Definition
//
class InformStyler
{
// Definitions
  private:
	enum	// Outer State
	{
		CommentFlag =			(1<<16),
		SingleQuoteFlag =		(1<<17),
		DoubleQuoteFlag =		(1<<18),
		StatementFlag =			(1<<19),
		AfterMarkerFlag =		(1<<20),
		HighlightFlag =			(1<<21),
		HighlightAllFlag =		(1<<22),
		ColourBacktrackFlag =	(1<<23),
		AfterRestartFlag =		(1<<24),
		WaitDirectiveFlag =		(1<<25),
		DontKnowFlag =			(1<<26)
	};

	enum { InitialState = WaitDirectiveFlag };

// Interface
  public:
	InformStyler() :	m_State(InitialState)	{ }
	DWORD	ColourString(LPCTSTR text, unsigned char * map);
	DWORD	GetState() const	{ return m_State; }

	static BOOL	isisym(char c)	{ return __iscsym(c) || '$'==c || '#'==c; }
	static BOOL	isisymf(char c)	{ return __iscsymf(c) || '$'==c || '#'==c; }

// Implementation
  private:
	void	RunInnerState(LPCTSTR text, size_t textLength, size_t & pos);
	BOOL	MatchesKeyword(const LPCTSTR * table, LPCTSTR text);
	BOOL	Includes(DWORD mask) const;
	void	Set(DWORD mask);
	void	Clear(DWORD mask);

	BOOL	IsKeywordTerminal() const;
	BOOL	IsPointer() const;
	BOOL	IsAsterisk() const;
	BOOL	IsTerminal() const;
	WORD	InnerState() const;

	void	ClearInnerState();
	void	SetInnerState(WORD newIS);

  private:
	DWORD	m_State;

	static const LPCTSTR	s_CodeKeys[];
	static const LPCTSTR	s_ForegroundKeys[];
};


#endif //!defined(INFORMSTYLER_H)
