/*=========================================================================*\
|* Output.c
\*=========================================================================*/

#include "Gal2CC.h"
#include <float.h>
#include <limits.h>


/***************************************************************************\
* Data
*
*/
static FILE * lastFile = NULL;

static char lastFillStyle[256];
static char lastLayer[256];
static char lastLineStyle[256];
static char lastTextFont[256];

static int lastColour = INT_MAX;
static int lastTextJustify = INT_MAX;
static int lastTextStyle = INT_MAX;
static double lastLineWidth = DBL_MAX;
static double lastTextAngle = DBL_MAX;
static double lastTextHeight = DBL_MAX;


/***************************************************************************\
* Update*()
\***************************************************************************/

static BOOL UpdateLineStyle(FileRecord output, const char * lineStyle )
{
	if ( NULL == output.file || NULL == lineStyle )
		return FALSE;
	if ( strcmp(lastLineStyle, lineStyle) )
	{
		strncpy(lastLineStyle, lineStyle, 255);
		if ( 0 > fprintf(output.file, "LSTYLE %s\n", lineStyle) )
		{
			ReportWriteError(output, NULL);
			return FALSE;
		}
	}
	return TRUE;
}


static BOOL UpdateFillStyle(FileRecord output, const char * fillStyle )
{
	if ( NULL == output.file || NULL == fillStyle )
		return FALSE;
	if ( strcmp(lastFillStyle, fillStyle) )
	{
		strncpy(lastFillStyle, fillStyle, 255);
		if ( 0 > fprintf(output.file, "FSTYLE %s\n", fillStyle) )
		{
			ReportWriteError(output, NULL);
			return FALSE;
		}
	}
	return TRUE;
}


static BOOL UpdateTextFont(FileRecord output, const char * textFont)
{
	if ( NULL == output.file || NULL == textFont )
		return FALSE;
	if ( strcmp(lastTextFont, textFont) )
	{
		strncpy(lastTextFont, textFont, 255);
		if ( 0 > fprintf(output.file, "TSPECF %s\n", textFont) )
		{
			ReportWriteError(output, NULL);
			return FALSE;
		}
	}
	return TRUE;
}


static BOOL UpdateColour(FileRecord output, int colour)
{
	if ( NULL == output.file )
		return FALSE;
	if ( lastColour != colour )
	{
		lastColour = colour;
		if ( 0 > fprintf(output.file, "COLOR %d\n", colour) )
		{
			ReportWriteError(output, NULL);
			return FALSE;
		}
	}
	return TRUE;
}


static BOOL UpdateTextJustify(FileRecord output, int textJustify)
{
	if ( NULL == output.file )
		return FALSE;
	if ( lastTextJustify != textJustify )
	{
		lastTextJustify = textJustify;
		if ( 0 > fprintf(output.file, "TSPECJ %d\n", textJustify) )
		{
			ReportWriteError(output, NULL);
			return FALSE;
		}
	}
	return TRUE;
}


static BOOL UpdateTextStyle(FileRecord output, int textStyle)
{
	if ( NULL == output.file )
		return FALSE;
	if ( lastTextStyle != textStyle )
	{
		lastTextStyle = textStyle;
		if ( 0 > fprintf(output.file, "TSPECS %d\n", textStyle) )
		{
			ReportWriteError(output, NULL);
			return FALSE;
		}
	}
	return TRUE;
}


static BOOL UpdateLineWidth(FileRecord output, double lineWidth)
{
	if ( NULL == output.file )
		return FALSE;
	if ( lastLineWidth != lineWidth )
	{
		lastLineWidth = lineWidth;
		if ( 0 > fprintf(output.file, "LWIDTH %f\n", lineWidth) )
		{
			ReportWriteError(output, NULL);
			return FALSE;
		}
	}
	return TRUE;
}


static BOOL UpdateTextAngle(FileRecord output, double textAngle)
{
	if ( NULL == output.file )
		return FALSE;
	if ( lastTextAngle != textAngle )
	{
		lastTextAngle = textAngle;
		if ( 0 > fprintf(output.file, "TSPECA %f\n", textAngle) )
		{
			ReportWriteError(output, NULL);
			return FALSE;
		}
	}
	return TRUE;
}


static BOOL UpdateTextHeight(FileRecord output, double textHeight)
{
	if ( NULL == output.file )
		return FALSE;
	if ( lastTextHeight != textHeight )
	{
		lastTextHeight = textHeight;
		if ( 0 > fprintf(output.file, "TSPECH %f\n", textHeight) )
		{
			ReportWriteError(output, NULL);
			return FALSE;
		}
	}
	return TRUE;
}


/***************************************************************************\
* UpdateOutput()
*
*/
static BOOL UpdateOutput(FileRecord output)
{
	if ( output.file != lastFile )
	{
		lastFile = output.file;

		lastFillStyle[0] = '\0';
		lastLineStyle[0] = '\0';
		lastTextFont[0] = '\0';

		lastColour = INT_MAX;
		lastTextJustify = INT_MAX;
		lastTextStyle = INT_MAX;
		lastLineWidth = DBL_MAX;
		lastTextAngle = DBL_MAX;
		lastTextHeight = DBL_MAX;

		strcpy(lastLayer, "STANDARD");
		if ( 0 > fprintf(output.file, "LAYER STANDARD\n") )
		{
			ReportWriteError(output, NULL);
			return FALSE;
		}
	}
	return TRUE;
}


/***************************************************************************\
* ArcTangent()
*
*/
double ArcTangent(DoubleCoord start, DoubleCoord end)
{
	double angle;
	double xdiff = end.x - start.x;
	double ydiff = end.y - start.y;
	if ( xdiff )
	{
		angle = atan(ydiff/xdiff);
		if ( 0 > xdiff )
		{
			if ( 0 > ydiff )
				angle -= PI;
			else
				angle += PI;
		}
	}
	else
		angle = (0 > ydiff) ? -HALF_PI : HALF_PI;
	return angle;
}


/***************************************************************************\
* BeginLayer()
*
*/
void BeginLayer(FileRecord output, const char * layer)
{
	if ( NULL == output.file || NULL == layer || !UpdateOutput(output) )
		return;
	if ( strcmp(lastLayer, layer) )
	{
		strncpy(lastLayer, layer, 255);
		if ( 0 > fprintf(output.file, "LAYER %s\n", layer) )
			ReportWriteError(output, NULL);
	}
}


/***************************************************************************\
* EndLayer()
*
*/
void EndLayer(FileRecord output)
{
	lastFile = NULL;
}


/***************************************************************************\
* OutputHeader()
*
*/
void OutputHeader(FileRecord output)
{
	UpdateOutput(output);
}


/***************************************************************************\
* OutputArc()
*
*/
void OutputArc(FileRecord output, double x, double y, double radius, 
			   double startAngle, double endAngle, int colour, 
			   double lineWidth, const char * lineStyle, 
			   const char * fillStyle)
{
	if ( NULL == output.file || NULL == lineStyle || NULL == fillStyle ||
		 !UpdateOutput(output) || 
		 !UpdateColour(output, colour) ||
		 !UpdateLineWidth(output, lineWidth) || 
		 !UpdateLineStyle(output, lineStyle) ||
		 !UpdateFillStyle(output, fillStyle) )
		return;
	if ( 0 > fprintf(output.file, "ARCS %f,%f;<%f,%f;%f\n",
					 x, y, startAngle, radius, endAngle) )
		ReportWriteError(output, NULL);
}


/***************************************************************************\
* OutputCircle()
*
*/
void OutputCircle(FileRecord output, double x, double y, double radius, 
				  int colour, double lineWidth, const char * lineStyle, 
				  const char * fillStyle)
{
	if ( NULL == output.file || NULL == lineStyle || NULL == fillStyle ||
		 !UpdateOutput(output) || 
		 !UpdateColour(output, colour) ||
		 !UpdateLineWidth(output, lineWidth) || 
		 !UpdateLineStyle(output, lineStyle) ||
		 !UpdateFillStyle(output, fillStyle) )
		return;
	if ( 0 > fprintf(output.file, "CIRR %f;%f,%f;\n", radius, x, y) )
		ReportWriteError(output, NULL);
}


/***************************************************************************\
* OutputLineWithGap()
*
*/
void OutputLineWithGap(FileRecord output, DoubleCoord start, DoubleCoord end, 
					   double gapRadius, int colour, double lineWidth, 
					   const char * lineStyle, const char * fillStyle)
{
	double angle, reverseAngle;

	if ( NULL == output.file || NULL == lineStyle || NULL == fillStyle ||
		 !UpdateOutput(output) || 
		 !UpdateColour(output, colour) ||
		 !UpdateLineWidth(output, lineWidth) || 
		 !UpdateLineStyle(output, lineStyle) ||
		 !UpdateFillStyle(output, fillStyle) )
		return;

	/* Having trouble with atan2, so do it myself: */
	angle = RadiansToDegrees(ArcTangent(start, end));
	reverseAngle = ( 0 > angle ) ? angle + 180 : angle - 180;

	if ( 0 > fprintf(output.file, "LINE REF %f,%f;<%f,%f;REF %f,%f;<%f,%f;\n",
					 start.x, start.y, angle, gapRadius, 
					 end.x, end.y, reverseAngle, gapRadius) )
		ReportWriteError(output, NULL);
}


/***************************************************************************\
* OutputSymbol()
*
*/
void OutputSymbol(FileRecord output, const char * symbol, double x, double y, 
				  double hscale, double vscale, double rotation)
{
	if ( NULL == output.file || NULL == symbol || !UpdateOutput(output) )
		return;
	if ( 0 > fprintf(output.file, "INSSYM %s\n%f;%f;%f;%f,%f;\n",
					 symbol, hscale, vscale, rotation, x, y) )
		ReportWriteError(output, NULL);
}


/***************************************************************************\
* OutputText()
*
*/
void OutputText(FileRecord output, const char * text, int length, 
				double x, double y, int colour, const char * font, 
				double height, double angle, int justify, int style)
{
	if ( NULL == output.file || NULL == text || !UpdateOutput(output) || 
		 !UpdateColour(output, colour) ||
		 !UpdateTextFont(output, font) || 
		 !UpdateTextHeight(output, height) ||
		 !UpdateTextAngle(output, angle) ||
		 !UpdateTextJustify(output, justify) ||
		 !UpdateTextStyle(output, style) )
		return;
	if ( 0 == length )
		length = strlen(text);
	if ( 0 > fprintf(output.file, "TEXTM %.*s\n%f,%f\n", length, text, x, y) )
		ReportWriteError(output, NULL);
}


/***************************************************************************\
* BeginOutput()
*
*/
BOOL BeginOutput(const char * givenFileName, FileRecord * output)
{
	if ( OpenOutputFile(givenFileName, output) )
	{
		if ( 0 <= fprintf(output->file, "ECOFF\n") )
			return TRUE;
		CloseOutputFile(*output);
	}
	return FALSE;
}


/***************************************************************************\
* EndOutput()
*
*/
void EndOutput(FileRecord file)
{
	fprintf(file.file, "ECON\n");
	CloseOutputFile(file);
}


