/*=========================================================================*\
|* TradeRoute.c
\*=========================================================================*/

#include "Gal2CC.h"

#define RIGHT_EDGE_X			485.0
#define TOP_EDGE_Y				701.480577

#define	SECTOR_EDGE_CUTOFF_X	(485.0+1.0)
#define	SECTOR_EDGE_CUTOFF_Y	(701.480577+1.0)

#define	SUBSECTOR_EDGE_CUTOFF_X	(125.0+1.0)
#define	SUBSECTOR_EDGE_CUTOFF_Y	(181.86534+1.0)

#define TRADE_ROUTE_EXCLUSION	5
#define TRADE_ROUTE_LINE_WIDTH	0.75
#define TRADE_ROUTE_LINE_STYLE	"Solid"
#define TRADE_ROUTE_FILL_STYLE	"Solid"

#define SubsectorX(hex)	(((hex.x-1)%SUBSECTOR_HEX_WIDTH)+1)
#define SubsectorY(hex)	(((hex.y-1)%SUBSECTOR_HEX_HEIGHT)+1)

#define SubsectorXBase(hex)	\
	(((hex.x-1)/SUBSECTOR_HEX_WIDTH)*SUBSECTOR_HEX_WIDTH)
#define SubsectorYBase(hex)	\
	(((hex.y-1)/SUBSECTOR_HEX_HEIGHT)*SUBSECTOR_HEX_HEIGHT)


/***************************************************************************\
* Data
*
*/
static BOOL tradeRoutePrintingMode = TRUE;

static const int colourMap[16] =
{
	0,	/* Black */
	7,	/* Blue */
	13,	/* Green */
	9,	/* Cyan */
	11,	/* Red */
	25,	/* Magenta */
	10,	/* Brown */
	16,	/* White */
	14,	/* Dark Grey */
	3,	/* Light Blue */
	1,	/* Light Green */
	5,	/* Light Cyan */
	2,	/* Light Red */
	6,	/* Light Magenta */
	4,	/* Light Yellow */
	15	/* Bright White */
};


/***************************************************************************\
* SetTradeRoutePrinting()
*
*/
void SetTradeRoutePrinting(BOOL mode)	{ tradeRoutePrintingMode = mode; }


/***************************************************************************\
* OutputTradeRoute()
*
*/
BOOL OutputTradeRoute(FileRecord output, const char * tradeLine, 
					  int subsectorIndex, BOOL subsectorOnly)
{
	DoubleCoord end[2];
	IntCoord intEnd[2];
	IntCoord subsectorOffset;
	int colour;
	int count;
	BOOL offLeft = FALSE;
	BOOL offRight = FALSE;
	BOOL offTop = FALSE;
	BOOL offBottom = FALSE;

	if ( !tradeRoutePrintingMode )
		return TRUE;

	if ( NULL == tradeLine || '\0' == *tradeLine )
		return FALSE;

	if ( 0 > subsectorIndex || 16 <= subsectorIndex )
		return FALSE;

	count = sscanf(&tradeLine[1], " %2d%2d %2d%2d %d %d %d", 
				   &intEnd[0].x, &intEnd[0].y, &intEnd[1].x, &intEnd[1].y, 
				   &subsectorOffset.x, &subsectorOffset.y, &colour);
	if ( 7 > count )
	{
		if ( 6 > count )
		{
			if ( 5 > count )
			{
				if ( 4 > count )
					return FALSE;
				subsectorOffset.x = 0;
			}
			subsectorOffset.y = 0;
		}
		colour = 5;
	}
	else if ( 0 > colour || 15 < colour )
		colour = 5;
	else
		colour = colourMap[colour];

	/* Adjust first end of the trade route 
	   (Galactic sometimes has it in the wrong subsector) */
	intEnd[0].x = ( (intEnd[0].x - 1) % SUBSECTOR_HEX_WIDTH ) + 1 +
				  SUBSECTOR_HEX_WIDTH * (subsectorIndex % 4);
	intEnd[0].y = ( (intEnd[0].y - 1) % SUBSECTOR_HEX_HEIGHT ) + 1 +
				  SUBSECTOR_HEX_HEIGHT * (subsectorIndex / 4);

	/* Generate my own offsets (Galactic often gets it wrong) 
	   Assume no routes longer than half a sector in x or y dirs */
	subsectorOffset.x = (intEnd[1].x-1) / SUBSECTOR_HEX_WIDTH -
						(intEnd[0].x-1) / SUBSECTOR_HEX_WIDTH;
	while ( -2 > subsectorOffset.x )
		subsectorOffset.x += 4;
	while ( 2 < subsectorOffset.x )
		subsectorOffset.x -= 4;

	subsectorOffset.y = (intEnd[1].y-1) / SUBSECTOR_HEX_HEIGHT -
						(intEnd[0].y-1) / SUBSECTOR_HEX_HEIGHT;
	while ( -2 > subsectorOffset.y )
		subsectorOffset.y += 4;
	while ( 2 < subsectorOffset.y )
		subsectorOffset.y -= 4;

	/* Handle offsets (only required if go outside (sub)sector) */
	if ( 0 > subsectorOffset.x )		/* Maybe off left */
	{
		if ( subsectorOnly ||
			 0 >= intEnd[0].x + SUBSECTOR_HEX_WIDTH * subsectorOffset.x )
		{
			intEnd[1].x = SubsectorXBase(intEnd[0]) + SubsectorX(intEnd[1]) + 
						  SUBSECTOR_HEX_WIDTH * subsectorOffset.x;
			offLeft = TRUE;
		}
	}
	else if ( 0 < subsectorOffset.x )	/* Maybe off right */
	{
		if ( subsectorOnly ||
			 SECTOR_HEX_WIDTH < intEnd[0].x + SUBSECTOR_HEX_WIDTH * subsectorOffset.x )
		{
			intEnd[1].x = SubsectorXBase(intEnd[0]) + SubsectorX(intEnd[1]) + 
						  SUBSECTOR_HEX_WIDTH * subsectorOffset.x;
			offRight = TRUE;
		}
	}
	if ( 0 > subsectorOffset.y )		/* Maybe off top */
	{
		if ( subsectorOnly ||
			 0 >= intEnd[0].y + SUBSECTOR_HEX_HEIGHT * subsectorOffset.y )
		{
			intEnd[1].y = SubsectorYBase(intEnd[0]) + SubsectorY(intEnd[1]) + 
						  SUBSECTOR_HEX_HEIGHT * subsectorOffset.y;
			offTop = TRUE;
		}
	}
	else if ( 0 < subsectorOffset.y )	/* Maybe off bottom */
	{
		if ( subsectorOnly ||
			 SECTOR_HEX_HEIGHT < intEnd[0].y + SUBSECTOR_HEX_HEIGHT * subsectorOffset.y )
		{
			intEnd[1].y = SubsectorYBase(intEnd[0]) + SubsectorY(intEnd[1]) + 
						  SUBSECTOR_HEX_HEIGHT * subsectorOffset.y;
			offBottom = TRUE;
		}
	}

	if ( ( intEnd[0].x == intEnd[1].x && intEnd[0].y == intEnd[1].y ) ||
		 !HexNumberToCoord(intEnd[0], &end[0], FALSE) || 
		 !HexNumberToCoord(intEnd[1], &end[1], FALSE) )
		return FALSE;

	if ( subsectorOnly && 
		 ( !AdjustRelativeToSubsector(&end[0], subsectorIndex) || 
		   !AdjustRelativeToSubsector(&end[1], subsectorIndex) ) )
		return FALSE;

	if ( offLeft || offRight || offTop || offBottom )
	{
		double xdiff = end[1].x - end[0].x;
		double ydiff = end[1].y - end[0].y;
		double distance = sqrt(xdiff*xdiff + ydiff*ydiff);
		double otherDistance;
		double angle = ArcTangent(end[0], end[1]);
		double edgeCutoffX = subsectorOnly ? SUBSECTOR_EDGE_CUTOFF_X : SECTOR_EDGE_CUTOFF_X;
		double edgeCutoffY = subsectorOnly ? SUBSECTOR_EDGE_CUTOFF_Y : SECTOR_EDGE_CUTOFF_Y;

		if ( offLeft )
			distance = fabs(end[0].x / cos(angle));
		else if ( offRight )
			distance = fabs(( edgeCutoffX - end[0].x ) / cos(angle));

		if ( offTop )
		{
			otherDistance = fabs(( edgeCutoffY - end[0].y ) / sin(angle));
			distance = min(distance, otherDistance);
		}
		else if ( offBottom )
		{
			otherDistance = fabs(end[0].y / sin(angle));
			distance = min(distance, otherDistance);
		}
		distance += TRADE_ROUTE_EXCLUSION;
		end[1].x = end[0].x + distance * cos(angle);
		end[1].y = end[0].y + distance * sin(angle);
	}
	OutputLineWithGap(output, end[0], end[1], TRADE_ROUTE_EXCLUSION,
					  colour, TRADE_ROUTE_LINE_WIDTH, 
					  TRADE_ROUTE_LINE_STYLE, TRADE_ROUTE_FILL_STYLE);
	return TRUE;
}

